package org.msh.tb.export_rest.cases;

import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.international.LocaleSelector;
import org.jboss.seam.international.Messages;
import org.msh.tb.entities.AdministrativeUnit;
import org.msh.tb.entities.UserWorkspace;
import org.msh.tb.entities.enums.*;
import org.msh.tb.export_rest.core.*;
import org.msh.tb.export_rest.dbreader.ReaderSQLQuery;
import org.msh.tb.export_rest.dbreader.SQLBlockReader;
import org.msh.tb.export_rest.core.BlockReader;
import org.msh.tb.export_rest.core.ExcelExportAsyncService;
import org.msh.tb.export_rest.core.ExcelExportException;
import org.msh.tb.ng.entities.enums.HIVPosition;
import org.msh.tb.ng.entities.enums.HIVPositionDetail;
import org.msh.tb.ng.entities.enums.IntakeAntiDrugsDuration;

import javax.persistence.EntityManager;
import java.io.File;
import java.io.IOException;
import java.util.*;

/**
 * Generate an Excel file with the cases based on the request. The Excel generation is
 * done asynchronously
 * <p>
 * Created by rmemoria on 13/7/17.
 */
@Name("casesExcelExportService")
@AutoCreate
@Scope(ScopeType.APPLICATION)
public class CasesExcelExportService {

    @In
    ExcelExportAsyncService excelExportAsyncService;


    /**
     * Start exporting the cases based on the given request. The exporting operation
     * will be done in a background task
     *
     * @param req
     * @return the task ID to get status and the file name with the exported cases
     */
    public String initExport(CaseExportRequest req) {
        try {
            LocaleSelector.instance().setLocaleString("en_NG");

            File file = File.createTempFile("cases", "xlsx");

            List<BlockReader> readers = new ArrayList<BlockReader>();
            addReader(readers, createTreatmentReader(req));
            addReader(readers, createComorbidityReader(req));
            addReader(readers, createSideEffectReader(req));
            addReader(readers, createXpertReader(req));
            addReader(readers, createMicroscopyReader(req));
            addReader(readers, createCultureReader(req));
            addReader(readers, createHivReader(req));
            addReader(readers, createDSTReader(req));
            addReader(readers, createXrayReader(req));
            addReader(readers, createMedicalExaminationReader(req));

            String taskId = excelExportAsyncService.startExport(file,
                    createCasesReader(req),
                    readers);
            return taskId;
        } catch (IOException e) {
            throw new ExcelExportException(e);
        }
    }

    private void addReader(List<BlockReader> readers, BlockReader reader) {
        if (reader != null) {
            readers.add(reader);
        }
    }

    /**
     * Create a reader to return the treatments
     *
     * @return
     */
    private BlockReader createTreatmentReader(CaseExportRequest req) {
        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("k.abbrevName", "Medicine")
                .from("tbcase", "a")
                .addJoin("(select pm.case_id, m.abbrevName\n" +
                        "from prescribedmedicine pm \n" +
                        "join medicine m on pm.medicine_id=m.id)", "k", "k.case_id=a.id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id")
                .setKeyField("a.id");

        // it is just one single column, so buffer can be bigger
        reader.setBufferSize(500);
        prepareRestrictions(reader.getQuery(), req);

        // calc number of columns
        List<String> titles = reader.getQuery().calcGroupedValues("k.abbrevName");

        return new PrefixedTitleReader(
                new ValueGroupBlockReader(reader, titles),
                "Treatment - ");
    }

    /**
     * Create reader for Xpert exam results
     *
     * @param req
     * @return
     */
    private BlockReader createXpertReader(CaseExportRequest req) {
        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("e.dateCollected", Messages.instance().get("PatientSample.dateCollected"))
                .addField("e.result", "Result", XpertResult.class)
                .from("examxpert", "e")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id, e.dateCollected")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        if (!req.isExamsXpert()) {
            reader.getQuery()
                    .addRestriction("e.id = (select id\n" +
                            "   from examxpert e2 where e2.case_id=e.case_id\n" +
                            "   order by e2.dateCollected limit 1)");
            reader.setBufferSize(500);

            return new PrefixedTitleReader(reader, "Xpert Diag. - ");
        }

        int maxColumns = reader.getQuery().calcMaxGroupCount();

        return new PrefixedTitleReader(
                new GroupedBlockReader(reader, maxColumns),
                "Xpert");
    }


    /**
     * Create reader to return the culture exams
     *
     * @param req
     * @return
     */
    private BlockReader createCultureReader(CaseExportRequest req) {
        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("e.dateCollected", Messages.instance().get("PatientSample.dateCollected"))
                .addField("e.result", "Result", CultureResult.class)
                .from("examculture", "e")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id, e.dateCollected")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        // if culture result should not return all record, just return the diagnosis
        if (!req.isExamsCulture()) {
            reader.getQuery().addRestriction("e.id = (select id\n" +
                    " from examculture e2 where e2.case_id=e.case_id\n" +
                    " order by e2.dateCollected limit 1)");
            reader.setBufferSize(500);

            return new PrefixedTitleReader(reader, "Culture Diag. - ");
        }

        int maxColumns = reader.getQuery().calcMaxGroupCount();

        return new PrefixedTitleReader(
                new GroupedBlockReader(reader, maxColumns),
                "Culture - ");
    }

    /**
     * Create reader for case side effects
     *
     * @param req
     * @return
     */
    private BlockReader createSideEffectReader(CaseExportRequest req) {
        if (!req.isSideEffect()) {
            return null;
        }

        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("se.name1", "Name")
                .addField("e.SE_MONTH", "Month")
                .from("casesideeffect", "e")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("fieldvalue", "se", "se.id = e.sideeffect_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("e.SE_MONTH")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        reader.setBufferSize(1000);

        int maxColumns = reader.getQuery().calcMaxGroupCount();

        return new PrefixedTitleReader(
                new GroupedBlockReader(reader, maxColumns),
                "Adverse Reaction - ");
    }


    /**
     * Create reader for microscopy exams
     *
     * @param req
     * @return
     */
    private BlockReader createMicroscopyReader(CaseExportRequest req) {
        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("e.dateCollected", Messages.instance().get("PatientSample.dateCollected"))
                .addField("e.result", "Result", MicroscopyResult.class)
                .from("exammicroscopy", "e")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id, e.dateCollected")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        reader.setBufferSize(1000);

        if (!req.isExamsMicroscopy()) {
            reader.getQuery().addRestriction("e.id = (select id\n" +
                    " from exammicroscopy e2 where e2.case_id=e.case_id\n" +
                    " order by e2.dateCollected limit 1)");

            return new PrefixedTitleReader(reader, "Microscopy Diag. - ");
        }

        int maxColumns = reader.getQuery().calcMaxGroupCount();

        return new PrefixedTitleReader(
                new GroupedBlockReader(reader, maxColumns),
                "Microscopy - ");
    }


    /**
     * Create reader to return X-ray records
     *
     * @param req
     * @return
     */
    private BlockReader createXrayReader(CaseExportRequest req) {
        if (!req.isExamsXray()) {
            return null;
        }

        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("e.event_date", "Date")
                .addField("f.name1", "Presentation")
                .addField("e.evolution", "Evolution", XRayEvolution.class)
                .from("examxray", "e")
                .addLeftJoin("fieldvalue", "f", "f.id = e.presentation_id")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id, e.event_date")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);
        int maxColumns = reader.getQuery().calcMaxGroupCount();

        return new PrefixedTitleReader(
                new GroupedBlockReader(reader, maxColumns),
                "X ray - ");
    }

    /**
     * Create reader for HIV results
     *
     * @param req
     * @return
     */
    private BlockReader createHivReader(CaseExportRequest req) {
        SQLBlockReader reader = new SQLBlockReader();

        if (!req.isExamsHIV()) {
            reader.getQuery()
                    .addField("'Result'", "Result")
                    .addField("e.result", "Result", HIVResult.class)
                    .from("examhiv", "e")
                    .addJoin("tbcase", "a", "a.id = e.case_id")
                    .addJoin("patient", "p", "p.id = a.patient_id")
                    .orderBy("a.id, e.result desc")
                    .setKeyField("a.id");

            reader.setBufferSize(800);
            prepareRestrictions(reader.getQuery(), req);

            String[] cols = {"Result"};

            return new PrefixedTitleReader(new ValueGroupBlockReader(reader, Arrays.asList(cols), 1), "HIV - ");
        }

        reader.getQuery()
                .addField("e.event_date", "Date")
                .addField("e.result", "Result", HIVResult.class)
                .from("examhiv", "e")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id, e.event_date")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        reader.setBufferSize(500);

        int maxColumns = reader.getQuery().calcMaxGroupCount();

        return new PrefixedTitleReader(
                new GroupedBlockReader(reader, maxColumns),
                "Teste HIV - ");
    }

    /**
     * Create reader for DST exams
     *
     * @param req
     * @return
     */
    private BlockReader createDSTReader(CaseExportRequest req) {
        if (!req.isExamsDST()) {
            return null;
        }

        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("s.name1", "Substance")
                .addField("r.result", "Result", DstResult.class)
                .from("examdstresult", "r")
                .addJoin("substance", "s", "s.id=r.substance_id")
                .addJoin("examdst", "e", "e.id = r.exam_id")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .addRestriction("e.dateCollected <= a.diagnosisDate")
                .orderBy("a.id, r.result desc")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        reader.setBufferSize(200);

        List<String> columns = reader.getQuery().calcGroupedValues("s.name1");

        return new PrefixedTitleReader(
                new ValueGroupBlockReader(reader, columns, 1),
                "DST - ");
    }


    private BlockReader createComorbidityReader(CaseExportRequest req) {
        if (!req.isComorbidities()) {
            return null;
        }

        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("f.name1", "Name")
                .from("casecomorbidity", "r")
                .addJoin("fieldvalue", "f", "f.id=r.comorbidity_id")
                .addJoin("tbcase", "a", "a.id = r.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id, f.name1")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        List<String> columns = reader.getQuery().calcGroupedValues("f.name1");

        return new PrefixedTitleReader(new ValueGroupBlockReader(reader, columns), "Comorbidity - ");
    }


    /**
     * Create reader to return the medical examinations
     * @param req
     * @return
     */
    private BlockReader createMedicalExaminationReader(CaseExportRequest req) {
        if (!req.isMedicalExaminations()) {
            return null;
        }

        Map<String, String> msgs = Messages.instance();

        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .addField("e.event_date", "Date")
                .addField("e.weight", "Peso (Kg)")
                .addField("e.height", msgs.get("MedicalExamination.height"))
                .from("medicalexamination", "e")
                .addJoin("tbcase", "a", "a.id = e.case_id")
                .addJoin("patient", "p", "p.id = a.patient_id")
                .orderBy("a.id, e.event_date")
                .setKeyField("a.id");

        prepareRestrictions(reader.getQuery(), req);

        int maxGroupCols = reader.getQuery().calcMaxGroupCount();

        return new PrefixedTitleReader(
                new GroupedBlockReader(reader, maxGroupCols),
                "Medical Examination - ");
    }

    /**
     * Create the main reader that will return case data
     *
     * @param req
     * @return
     */
    private BlockReader createCasesReader(CaseExportRequest req) {
        Map<String, String> msgs = Messages.instance();

        SQLBlockReader reader = new SQLBlockReader();

        reader.getQuery()
                .from("tbcase", "a")
                .addJoin("tbcaseng", "b", "b.id = a.id")
                .addJoin("patient", "p", "p.id = a.patient_id")

                .addJoin("tbunit", "u", "u.id = a.owner_unit_id")
                .addJoin("administrativeunit", "au1", "au1.id = u.adminunit_id")
                .addLeftJoin("administrativeunit", "au2", "au2.id=au1.parent_id")
                .addLeftJoin("administrativeunit", "au3", "au3.id=au2.parent_id")

                .addJoin("tbunit", "notiftbunit", "notiftbunit.id = a.NOTIFICATION_UNIT_ID")
                .addJoin("administrativeunit", "notifau1", "notifau1.id = notiftbunit.adminunit_id")
                .addLeftJoin("administrativeunit", "notifau2", "notifau2.id=notifau1.parent_id")
                .addLeftJoin("administrativeunit", "notifau3", "notifau3.id=notifau2.parent_id")

                .addLeftJoin("regimen", "r", "r.id=a.regimen_id")
                .addLeftJoin("regimen", "inir", "inir.id=a.regimen_ini_id")
                .addLeftJoin("fieldvalue", "pt", "pt.id=a.pulmonary_id")
                .addLeftJoin("fieldvalue", "ept", "ept.id=a.extrapulmonary_id")
                .addLeftJoin("fieldvalue", "ept2", "ept2.id=a.extrapulmonary2_id")
                .addLeftJoin("administrativeunit", "notifaddau", "notifaddau.id=a.NOTIF_ADMINUNIT_ID")
                .addLeftJoin("administrativeunit", "notifaddau2", "notifaddau2.id=notifaddau.parent_id")
                .addLeftJoin("administrativeunit", "notifaddau3", "notifaddau3.id=notifaddau2.parent_id")

                .addLeftJoin("fieldvalue", "sourceReferral", "sourceReferral.id=b.SOURCEREFERRAL_ID")
                .addLeftJoin("fieldvalue", "occupation", "occupation.id=b.occupation_id")
                .addLeftJoin("fieldvalue", "maritalstatus", "maritalstatus.id=b.maritalstatus_id")
                .addLeftJoin("fieldvalue", "dotProvider", "dotProvider.id=b.dotProvider_id")
                .orderBy("a.id")
                .setKeyField("a.id");

        reader.getQuery()
                .addField("p.recordNumber", msgs.get("Patient.recordNumber"))
                .addField("a.caseNumber", "Case number")
                .addField("b.tbRegistrationNumber", msgs.get("TbCase.TBRegistrationNumber"))
                .addField("p.lastName", msgs.get("Patient.lastName"))
                .addField("p.middleName", msgs.get("Patient.middleName"))
                .addField("p.PATIENT_NAME", msgs.get("Patient.firstname"))
                .addField("p.motherName", msgs.get("Patient.motherName"))
                .addField("a.diagnosisType", msgs.get("DiagnosisType"), DiagnosisType.class)
                .addField("a.classification", msgs.get("CaseClassification"), CaseClassification.class)
                .addField("a.state", "Case state / outcome", CaseState.class)
                .addField("p.gender", msgs.get("Gender"), Gender.class)
                .addField("p.birthDate", "Birth date")
                .addField("a.age", "Age")
                .addField("a.nationality", msgs.get("Nationality"), Nationality.class)
                .addField("a.CURR_ADDRESS", msgs.get("cases.details.addressnotif"))
                .addField("a.CURR_COMPLEMENT", msgs.get("Address.complement"))
                .addField("notifaddau.name1", "Curr. Addr. - Admin unit")
                .addField("notifaddau2.name1", "Curr. Addr. - Admin unit")
                .addField("notifaddau3.name1", "Curr. Addr. - Admin unit")
                .addField("a.CURR_ZIPCODE", msgs.get("Address.zipCode"))
                .addField("a.phoneNumber", msgs.get("TbCase.phoneNumber"))
                .addField("a.mobileNumber", msgs.get("TbCase.mobileNumber"))
                .addField("b.emailAddress", msgs.get("TbCase.emailAddress"))
                .addField("notiftbunit.name1", msgs.get("TbCase.notificationUnit"))
                .addField("notifau1.name1", "Registration unit - Admin unit")
                .addField("notifau2.name1", "Registration unit - Admin unit")
                .addField("notifau3.name1", "Registration unit - Admin unit")
                .addField("u.name1", "Treatment unit")
                .addField("au1.name1", "Treatment unit - Admin unit")
                .addField("au2.name1", "Treatment unit - Admin unit")
                .addField("au3.name1", "Treatment unit - Admin unit")
                .addField("a.registrationDate", msgs.get("TbCase.registrationDate"))
                .addField("r.regimen_name", msgs.get("Regimen"))
                .addField("inir.regimen_name", "Initial regimen")
                .addField("a.diagnosisDate", msgs.get("TbCase.diagnosisDate"))
                .addField("a.iniTreatmentDate", msgs.get("TbCase.iniTreatmentDate"))
                .addField("a.endTreatmentDate", msgs.get("TbCase.endTreatmentDate"))
                .addField("a.drugResistanceType", msgs.get("DrugResistanceType"), DrugResistanceType.class)
                .addField("a.infectionSite", msgs.get("InfectionSite"), InfectionSite.class)
                .addField("pt.name1", msgs.get("TbField.PULMONARY_TYPES"))
                .addField("ept.name1", msgs.get("TbField.EXTRAPULMONARY_TYPES"))
                .addField("ept2.name1", msgs.get("TbField.EXTRAPULMONARY_TYPES"))
                .addField("a.patientType", msgs.get("PatientType"), PatientType.class)
                .addField("a.treatmentCategory", "Treatment Category", TreatmentCategory.class)
                .addField("a.caseDefinition", "Case Definition", CaseDefinition.class)
                .addField("sourceReferral.name1", msgs.get("TbField.SOURCE_REFERRAL"))
                .addField("occupation.name1", msgs.get("TbField.OCCUPATION"))
                .addField("maritalstatus.name1", msgs.get("TbField.MARITAL_STATUS"))
                .addField("dotProvider.name1", msgs.get("TbField.DOT_PROVIDER"))
                .addField("b.hivPosition", "HIV Position", HIVPosition.class)
                .addField("b.hivPositionDetail", "HIV position detail", HIVPositionDetail.class)
                .addField("b.intakeAntiTBDrugs", msgs.get("cases.historyAntiTBdrugs"), YesNoType.class)
                .addField("b.intakeAntiTBDrugsDuration", msgs.get("cases.durationweeks"), IntakeAntiDrugsDuration.class);

        prepareRestrictions(reader.getQuery(), req);

        reader.setBufferSize(600);

        return reader;
    }

    /**
     * Apply restrictions to the query based on the request. These restrictions are applied to
     * all queries used during exporting
     *
     * @param query
     * @param req
     */
    private void prepareRestrictions(ReaderSQLQuery query, CaseExportRequest req) {
        // include case classification restriction
        if (req.getClassification() != null) {
            query.addRestriction("a.classification = :cla");
            query.addParameter("cla", req.getClassification());
        }

        applyUnitRestrictions(query, req);
        applyDateRestrictions(query, req);
    }


    private void applyUnitRestrictions(ReaderSQLQuery query, CaseExportRequest req) {
        UserWorkspace uw = (UserWorkspace) Component.getInstance("userWorkspace");

        Integer adminUnitId;
        Integer unitId;

        switch (uw.getView()) {
            case TBUNIT:
                unitId = uw.getTbunit().getId();
                adminUnitId = null;
                break;
            case ADMINUNIT:
                adminUnitId = req.getUnitId() != null ? null : uw.getAdminUnit().getId();
                unitId = req.getUnitId();
                break;
            default:
                adminUnitId = req.getAdminUnitId();
                unitId = req.getUnitId();
        }

        // TB unit was set ?
        if (unitId != null) {
            query.addRestriction("a.owner_unit_id = :unitid");
            query.addParameter("unitid", unitId);
        } else if (adminUnitId != null) {
            // administrative unit was set ?
            EntityManager em = (EntityManager) Component.getInstance("entityManager");

            AdministrativeUnit au = em.find(AdministrativeUnit.class, adminUnitId);

            if (au != null) {
                if (!query.isTableAliasInUse("u")) {
                    query.addJoin("tbunit", "u", "u.id = a.owner_unit_id");
                }

                if (!query.isTableAliasInUse("au1")) {
                    query.addJoin("administrativeunit", "au1", "au1.id = u.adminunit_id");
                }

                query.addRestriction("au1.code like :aucode");
                query.addParameter("aucode", au.getCode() + "%");
            }
        } else {
            query.addRestriction("p.workspace_id = :workspaceId");
            query.addParameter("workspaceId", uw.getWorkspace().getId());
        }
    }

    /**
     * Apply restrictions according to the user view mixed with the unit and admin unit selected
     * by the user
     * @param query the query to receive the restrictions
     * @param req the request made by the user
     */
    private void applyDateRestrictions(ReaderSQLQuery query, CaseExportRequest req) {
        // check date periods
        Date iniDate = req.getIniDate();
        Date endDate = req.getEndDate();

        if (iniDate == null && endDate == null) {
            return;
        }

        String dateField = "a." + req.getDateToFilter().getField();

        if (iniDate != null && endDate != null) {
            query.addRestriction(dateField + " between :dtini and :dtend");
            query.addParameter("dtini", iniDate);
            query.addParameter("dtend", endDate);
        } else {
            if (iniDate != null) {
                query.addRestriction(dateField + " >= :dtini");
                query.addParameter("dtini", iniDate);
            }

            if (endDate != null) {
                query.addRestriction(dateField + " <= :dtend");
                query.addParameter("dtend", endDate);
            }
        }
    }
}
